% Dynamic OCT methods for Fig. 2 and Vis. 1
%
% Biomedical Optics Express paper titled:
% Review of dynamic optical coherence tomography for intracellular motility: 
%    Signals, metrics, and their applications 
%
% Amy Josefsberg, June, 2025

function [MeanDOCTimage] = dOCT_compute(imgStack,dOCTfunc,ChunkSize,ExpBalance,BgSubtract,BgCoords,LIVthres)
% INPUTS:
%  imgStack - OCT image stack in time (3rd dimension), double
%             Note: code will not work well on uint types, convert to double
%  dOCTfunc - name of function to apply to the data (call using @dOCTfunc)
%      choices are: func_Var (variance)
%                   func_SD (standard deviation)
%                   func_NSD (normalized standard deviation)
%                   func_LIV (logarithmic intensity variation)
%                   func_motil ("M", motility amplitude)
%  ChunkSize - integer number of images to use for windowed analysis
%              Set to 50 for FFOCT data, 49 for SDOCT data
%  ExpBalance - boolean, whether to do exposure balancing
%               Set to true for FFOCT data, false for SDOCT data
%  BgSubract - boolean, whether to do background subtraction
%              Set to false for Fig. 2, true for Vis. 1
%  BgCoords - integer coordinates (upper left, lower right) to define bounding box
%             for computing the background level
%             Set to [[50,50];[300,300]] for Vis. 1
%  LIVthres - double, threshold value for LIV computation, set to 0 unless LIV is used
%             Set to 50% of the mean image value for Fig. 2 and Vis. 1
% OUTPUTS:
%  MeanDOCTimage - dOCT image computed from the input stack, double

    % get dimensions of input image stack
    [hei,wid,nimg] = size(imgStack); 

    % Exposure Balancing (only for FFOCT data)
    if ExpBalance 
        ExpVec = squeeze(mean(mean(imgStack,1),2));
        for i = 1:nimg
            imgStack(:,:,i) = imgStack(:,:,i)/ExpVec(i);
        end
    end
    
    % windowed processing
    ivec = 1:ChunkSize:nimg+1; % indices where chunks start
    nchunks = size(ivec,2) - 1; % number of chunks available
    dOCTimages = zeros(hei,wid,nchunks); % pre-allocate output images
    for i = 1:nchunks
        % process dOCT metric within each temporal chunk
        if LIVthres > 0 % LIV method requires a threshold
            dOCTimages(:,:,i) = dOCTfunc(imgStack(:,:,ivec(i):ivec(i+1)-1),LIVthres);
        else
            dOCTimages(:,:,i) = dOCTfunc(imgStack(:,:,ivec(i):ivec(i+1)-1));
        end
    end
    MeanDOCTimage = mean(dOCTimages,3); % average the result

    % Background subtraction, only used for Vis 1
    if BgSubtract 
        BgLevel = mean(mean(MeanDOCTimage(BgCoords(1,2):BgCoords(2,2),BgCoords(1,1):BgCoords(2,1))));
        MeanDOCTimage = MeanDOCTimage - BgLevel;
    end

end